// Performance optimizations
const isMobile = window.innerWidth <= 768;
const isReducedMotion = window.matchMedia('(prefers-reduced-motion: reduce)').matches;

// Cache frequently used DOM elements
const domCache = {
    navbar: null,
    hero: null,
    backToTopBtn: null,
    navMenu: null,
    policySections: null
};

// Initialize DOM cache
function initDOMCache() {
    domCache.navbar = document.querySelector('.navbar');
    domCache.hero = document.querySelector('.hero');
    domCache.backToTopBtn = document.getElementById('backToTop');
    domCache.navMenu = document.querySelector('.nav-menu');
    domCache.policySections = document.querySelectorAll('.policy-section-content');
}

// Throttle function for performance
function throttle(func, limit) {
    let inThrottle;
    return function() {
        const args = arguments;
        const context = this;
        if (!inThrottle) {
            func.apply(context, args);
            inThrottle = true;
            setTimeout(() => inThrottle = false, limit);
        }
    }
}

// Debounce function for performance
function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

// DOM Content Loaded
document.addEventListener('DOMContentLoaded', function() {
    // Initialize DOM cache first
    initDOMCache();
    
    // Initialize all functionality with performance considerations
    initNavigation();
    initSearch();
    if (!isReducedMotion) {
        initScrollEffects();
        initAnimations();
    }
    initBackToTop();
    initDarkMode();
    initHeroCards();
    enableHeroCardAnimation();
});

// Navigation functionality
function initNavigation() {
    // We're using mobile-menu-toggle instead of hamburger now
    const hamburger = document.getElementById('mobile-menu-toggle');
    const navMenu = document.querySelector('.nav-menu');
    const navLinks = document.querySelectorAll('.nav-link');

    // Leave mobile menu toggle to inline initMobileMenu to avoid double-binding

    // Smooth scrolling for navigation links
    navLinks.forEach(link => {
        link.addEventListener('click', function(e) {
            const href = this.getAttribute('href');
            
            // Only handle internal links (starting with #)
            if (href && href.startsWith('#')) {
                e.preventDefault();
                const targetSection = document.querySelector(href);
                
                if (targetSection) {
                    const offsetTop = targetSection.offsetTop - 80; // Account for fixed navbar
                    window.scrollTo({
                        top: offsetTop,
                        behavior: 'smooth'
                    });
                }
            }
            // External links will navigate normally
        });
    });

    // Navbar scroll effect with RAF optimization
    let navbarScrollTicking = false;
    
    function updateNavbar() {
        if (domCache.navbar) {
            if (window.scrollY > 50) {
                domCache.navbar.style.background = 'rgba(255, 255, 255, 0.98)';
                domCache.navbar.style.boxShadow = '0 2px 20px rgba(0, 0, 0, 0.15)';
            } else {
                domCache.navbar.style.background = 'rgba(255, 255, 255, 0.95)';
                domCache.navbar.style.boxShadow = '0 2px 20px rgba(0, 0, 0, 0.1)';
            }
        }
        navbarScrollTicking = false;
    }
    
    function handleNavbarScroll() {
        if (!navbarScrollTicking) {
            requestAnimationFrame(updateNavbar);
            navbarScrollTicking = true;
        }
    }
    
    window.addEventListener('scroll', handleNavbarScroll, { passive: true });
}

// Search functionality
function initSearch() {
    const searchInput = document.getElementById('searchInput');
    const searchBtn = document.getElementById('searchBtn');
    const policySections = document.querySelectorAll('.policy-section-content');
    
    // Store the current search query globally so we can reapply it when theme changes
    window.currentSearchQuery = '';

    if (searchInput && searchBtn) {
        // Search on button click
        searchBtn.addEventListener('click', function() {
            window.performSearch();
        });

        // Search on Enter key
        searchInput.addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                e.preventDefault();
                window.performSearch();
            }
        });

        // Keyboard navigation for search button
        searchBtn.addEventListener('keydown', function(e) {
            if (e.key === 'Enter' || e.key === ' ') {
                e.preventDefault();
                window.performSearch();
            }
        });

        // Real-time search as user types (debounced for performance)
        const debouncedSearch = debounce(function() {
            const query = searchInput.value.toLowerCase().trim();
            if (query.length > 2) {
                window.performSearch();
            } else if (query.length === 0) {
                clearSearch();
            }
        }, 300);
        
        searchInput.addEventListener('input', debouncedSearch);
    }

    // Make performSearch globally accessible
    window.performSearch = function() {
        if (!searchInput) return; // Guard clause
        
        const query = searchInput.value.toLowerCase().trim();
        if (query.length === 0) {
            clearSearch();
            return;
        }
        
        // Store the current search query globally
        window.currentSearchQuery = query;

        let foundResults = false;
        policySections.forEach(section => {
            const text = section.textContent.toLowerCase();
            
            if (text.includes(query)) {
                section.style.display = 'block';
                // Check for dark mode and use appropriate highlight colors for the section
                const isDarkMode = document.body.getAttribute('data-theme') === 'dark';
                section.style.background = isDarkMode ? '#451a03' : '#f0f7ff';
                section.style.borderLeft = `4px solid ${isDarkMode ? '#f59e0b' : '#1e3a8a'}`;
                
                // Highlight matching text
                highlightText(section, query);
                foundResults = true;
            } else {
                section.style.display = 'none';
            }
        });

        // Show no results message
        if (!foundResults) {
            showNoResultsMessage();
        } else {
            hideNoResultsMessage();
        }

        // Scroll to first result
        const firstVisible = document.querySelector('.policy-section-content[style*="block"]');
        if (firstVisible) {
            firstVisible.scrollIntoView({ behavior: 'smooth', block: 'center' });
        }
        
        // Add click handlers for highlighted sections to toggle them
        addSectionToggleHandlers();
    }

    // Make clearSearch globally accessible
    window.clearSearch = function() {
        policySections.forEach(section => {
            section.style.display = 'block';
            section.style.background = '';
            section.style.cursor = 'default';
            section.style.transform = 'none';
            
            // Set border color based on theme
            const isDarkMode = document.body.getAttribute('data-theme') === 'dark';
            section.style.borderLeft = `4px solid ${isDarkMode ? '#f59e0b' : '#667eea'}`;
            
            // Remove toggle handlers
            section.removeEventListener('click', toggleSection);
            
            removeHighlights(section);
        });
        hideNoResultsMessage();
        
        // Clear the stored search query
        window.currentSearchQuery = '';
    }
    
    // For backward compatibility
    function clearSearch() {
        window.clearSearch();
    }

    function highlightText(element, query) {
        const walker = document.createTreeWalker(
            element,
            NodeFilter.SHOW_TEXT,
            null,
            false
        );

        const textNodes = [];
        let node;
        while (node = walker.nextNode()) {
            textNodes.push(node);
        }

        textNodes.forEach(textNode => {
            const text = textNode.textContent;
            const regex = new RegExp(`(${query})`, 'gi');
            if (regex.test(text)) {
                // Check for dark mode and use appropriate highlight colors
                const isDarkMode = document.body.getAttribute('data-theme') === 'dark';
                const highlightBg = isDarkMode ? '#78350f' : '#fff3cd';
                const highlightColor = isDarkMode ? '#ffffff' : '#1e3a8a';
                
                const highlightedText = text.replace(regex, `<mark style="background: ${highlightBg}; color: ${highlightColor}; padding: 2px 4px; border-radius: 3px;">$1</mark>`);
                const span = document.createElement('span');
                span.innerHTML = highlightedText;
                textNode.parentNode.replaceChild(span, textNode);
            }
        });
    }

    // Make removeHighlights globally accessible
    window.removeHighlights = function(element) {
        if (!element) return; // Guard clause
        
        const marks = element.querySelectorAll('mark');
        marks.forEach(mark => {
            mark.outerHTML = mark.textContent;
        });
    }
    
    // Add toggle functionality for highlighted sections
    function addSectionToggleHandlers() {
        const highlightedSections = document.querySelectorAll('.policy-section-content[style*="block"]');
        
        highlightedSections.forEach(section => {
            // Remove existing click handlers to avoid duplicates
            section.removeEventListener('click', toggleSection);
            
            // Add click handler for toggle effect
            section.addEventListener('click', toggleSection);
            
            // Add visual indicator that section is clickable
            section.style.cursor = 'pointer';
            section.style.transition = 'all 0.3s ease';
        });
    }
    
    function toggleSection(event) {
        // Don't toggle if clicking on links or buttons
        if (event.target.tagName === 'A' || event.target.tagName === 'BUTTON') {
            return;
        }
        
        const section = event.currentTarget;
        const isDarkMode = document.body.getAttribute('data-theme') === 'dark';
        
        // Toggle the section visibility
        if (section.style.display === 'none') {
            section.style.display = 'block';
            section.style.background = isDarkMode ? '#451a03' : '#f0f7ff';
            section.style.borderLeft = `4px solid ${isDarkMode ? '#f59e0b' : '#1e3a8a'}`;
        } else {
            section.style.display = 'none';
        }
        
        // Add visual feedback
        section.style.transform = 'scale(0.98)';
        setTimeout(() => {
            section.style.transform = 'scale(1)';
        }, 150);
    }
    
    // For backward compatibility
    function removeHighlights(element) {
        window.removeHighlights(element);
    }

    function showNoResultsMessage() {
        let noResults = document.getElementById('no-results-message');
        if (!noResults) {
            noResults = document.createElement('div');
            noResults.id = 'no-results-message';
            // Check for dark mode and use appropriate colors for no results message
            const isDarkMode = document.body.getAttribute('data-theme') === 'dark';
            noResults.style.cssText = `
                text-align: center;
                padding: 3rem;
                background: ${isDarkMode ? '#451a03' : '#e6f0ff'};
                color: ${isDarkMode ? '#ffffff' : '#1e3a8a'};
                border-radius: 15px;
                margin: 2rem 0;
                border: 1px solid ${isDarkMode ? '#f59e0b' : '#3b82f6'};
                box-shadow: 0 4px 15px rgba(${isDarkMode ? '245, 158, 11, 0.3' : '30, 64, 175, 0.15'});
            `;
            noResults.innerHTML = `
                <i class="fas fa-search" style="font-size: 3rem; margin-bottom: 1rem; opacity: 0.5;"></i>
                <h3>No results found</h3>
                <p>Try searching with different keywords or check your spelling.</p>
            `;
            
            const firstPolicySection = document.querySelector('.policy-section');
            if (firstPolicySection) {
                firstPolicySection.appendChild(noResults);
            }
        }
    }

    function hideNoResultsMessage() {
        const noResults = document.getElementById('no-results-message');
        if (noResults) {
            noResults.remove();
        }
    }
}

// Scroll effects
function initScrollEffects() {
    // Intersection Observer for animations
    const observerOptions = {
        threshold: 0.1,
        rootMargin: '0px 0px -50px 0px'
    };

    const observer = new IntersectionObserver(function(entries) {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.style.opacity = '1';
                entry.target.style.transform = 'translateY(0)';
            }
        });
    }, observerOptions);

    // Observe elements for animation
    const animatedElements = document.querySelectorAll('.nav-card, .policy-section-content, .contact-method');
    animatedElements.forEach(el => {
        el.style.opacity = '0';
        el.style.transform = 'translateY(30px)';
        el.style.transition = 'opacity 0.6s ease, transform 0.6s ease';
        observer.observe(el);
    });

    // Parallax effect for hero section using requestAnimationFrame
    let latestScrollY = window.pageYOffset;
    let ticking = false;

    function updateParallax() {
        if (domCache.hero) {
            domCache.hero.style.transform = `translateY(${latestScrollY * 0.5}px)`;
        }
        ticking = false;
    }

    function onScroll() {
        latestScrollY = window.pageYOffset;
        if (!ticking) {
            window.requestAnimationFrame(updateParallax);
            ticking = true;
        }
    }

    window.addEventListener('scroll', onScroll, { passive: true });
    window.requestAnimationFrame(updateParallax);
}

// Animations
function initAnimations() {
    // Removed unused stats animation code

    // Floating cards animation
    const cards = document.querySelectorAll('.card');
    cards.forEach((card, index) => {
        card.style.animationDelay = `${index * 2}s`;
    });

    // Hero title displays immediately - no typing effect
}

// Removed unused animateCounter function

// Back to top button
function initBackToTop() {
    if (domCache.backToTopBtn) {
        // RAF-optimized scroll handler for performance
        let backToTopTicking = false;
        
        function updateBackToTop() {
            if (window.pageYOffset > 300) {
                domCache.backToTopBtn.classList.add('show');
            } else {
                domCache.backToTopBtn.classList.remove('show');
            }
            backToTopTicking = false;
        }
        
        function handleBackToTopScroll() {
            if (!backToTopTicking) {
                requestAnimationFrame(updateBackToTop);
                backToTopTicking = true;
            }
        }
        
        window.addEventListener('scroll', handleBackToTopScroll, { passive: true });

        domCache.backToTopBtn.addEventListener('click', function() {
            window.scrollTo({
                top: 0,
                behavior: 'smooth'
            });
        });

        // Keyboard navigation support
        domCache.backToTopBtn.addEventListener('keydown', function(e) {
            if (e.key === 'Enter' || e.key === ' ') {
                e.preventDefault();
                domCache.backToTopBtn.click();
            }
        });
    }
}

// Keyboard navigation
document.addEventListener('keydown', function(e) {
    // ESC key to close mobile menu
    if (e.key === 'Escape') {
        const hamburger = document.getElementById('mobile-menu-toggle');
        const navMenu = document.querySelector('.nav-menu');
        if (hamburger && navMenu) {
            hamburger.classList.remove('active');
            navMenu.classList.remove('show');
        }
    }
    
    // Ctrl/Cmd + K to focus search
    if ((e.ctrlKey || e.metaKey) && e.key === 'k') {
        e.preventDefault();
        const searchInput = document.getElementById('searchInput');
        if (searchInput) {
            searchInput.focus();
        }
    }
});

// Print functionality
function printPage() {
    window.print();
}

// Add print button functionality if needed
document.addEventListener('DOMContentLoaded', function() {
    const printButtons = document.querySelectorAll('.print-btn');
    printButtons.forEach(btn => {
        btn.addEventListener('click', printPage);
    });
});

// Error handling
window.addEventListener('error', function(e) {
    // Handle both error object and ErrorEvent types
    if (e instanceof ErrorEvent) {
        // Error logged silently for production
    } else {
        // Error logged silently for production
    }
    
    // Could send to analytics service - safely access properties
    if (typeof gtag !== 'undefined') {
        const errorMessage = (e.error && e.error.message) || e.message || 'Unknown error';
        gtag('event', 'exception', {
            'description': errorMessage,
            'fatal': false
        });
    }
});

// Theme Toggle - Simplified Version
function initDarkMode() {
    // Initializing theme toggle
    
    // Wait a bit for DOM to be ready
    setTimeout(() => {
        const themeToggle = document.querySelector('.theme-toggle');
        const body = document.body;
        const html = document.documentElement;
        
        if (!themeToggle) {
            // Theme toggle button not found
            return;
        }
        
        // Check for saved theme preference or default to light mode
        const currentTheme = localStorage.getItem('theme') || 'light';
        
        body.setAttribute('data-theme', currentTheme);
        html.setAttribute('data-theme', currentTheme);
        
        // Update icon based on current theme
        const icon = themeToggle.querySelector('i');
        if (icon) {
            icon.className = currentTheme === 'dark' ? 'fas fa-sun' : 'fas fa-moon';
        }
        
        // Simple click handler
        themeToggle.onclick = function(e) {
            e.preventDefault();
            e.stopPropagation();
            
            const currentTheme = body.getAttribute('data-theme');
            const newTheme = currentTheme === 'dark' ? 'light' : 'dark';
            
            body.setAttribute('data-theme', newTheme);
            html.setAttribute('data-theme', newTheme);
            localStorage.setItem('theme', newTheme);
            
            // Update icon
            if (icon) {
                icon.className = newTheme === 'dark' ? 'fas fa-sun' : 'fas fa-moon';
            }
            
            // Visual feedback
            this.style.transform = 'scale(0.9)';
            setTimeout(() => {
                this.style.transform = 'scale(1)';
            }, 100);
            
            // Update search highlighting if search is active
            updateSearchHighlighting();
        };
    }, 100);
}

// Function to update search highlighting when theme changes
function updateSearchHighlighting() {
    // If there's an active search, reapply it with the new theme colors
    if (window.currentSearchQuery && window.currentSearchQuery.length > 0) {
        // First clear any existing highlights and styling
        const policySections = document.querySelectorAll('.policy-section-content');
        
        // Reset all sections first
        policySections.forEach(section => {
            // Keep display state
            const wasDisplayed = section.style.display !== 'none';
            
            // Reset styles
            section.style.background = '';
            const isDarkMode = document.body.getAttribute('data-theme') === 'dark';
            section.style.borderLeft = `4px solid ${isDarkMode ? '#f59e0b' : '#667eea'}`;
            
            // Remove highlights
            window.removeHighlights(section);
            
            // Restore display state
            section.style.display = wasDisplayed ? 'block' : 'none';
        });
        
        // Then reapply the search with updated theme colors
        window.performSearch();
    }
}

// Initialize theme system
document.addEventListener('DOMContentLoaded', function() {
    initDarkMode();
});

// Also try to initialize immediately in case DOM is already loaded
if (document.readyState === 'loading') {
    // DOM still loading
} else {
    // DOM already loaded, initializing theme immediately
    initDarkMode();
}

// Hero Cards functionality
function initHeroCards() {
    const privacyCard = document.querySelector('.privacy-card');
    const refundCard = document.querySelector('.refund-card');
    const termsCard = document.querySelector('.terms-card');
    
    if (privacyCard) {
        privacyCard.addEventListener('click', function() {
            const targetSection = document.getElementById('privacy');
            if (targetSection) {
                const offsetTop = targetSection.offsetTop - 80;
                window.scrollTo({
                    top: offsetTop,
                    behavior: 'smooth'
                });
            }
        });
    }
    
    if (refundCard) {
        refundCard.addEventListener('click', function() {
            const targetSection = document.getElementById('refund');
            if (targetSection) {
                const offsetTop = targetSection.offsetTop - 80;
                window.scrollTo({
                    top: offsetTop,
                    behavior: 'smooth'
                });
            }
        });
    }
    
    if (termsCard) {
        termsCard.addEventListener('click', function() {
            const targetSection = document.getElementById('terms');
            if (targetSection) {
                const offsetTop = targetSection.offsetTop - 80;
                window.scrollTo({
                    top: offsetTop,
                    behavior: 'smooth'
                });
            }
        });
    }
}

// Enable desktop hero card animations via a class to increase selector specificity
function enableHeroCardAnimation() {
    const cardsContainer = document.querySelector('.floating-cards');
    if (!cardsContainer) return;

    function apply() {
        if (window.innerWidth >= 1024) {
            cardsContainer.classList.add('animate-cards');
        } else {
            cardsContainer.classList.remove('animate-cards');
        }
    }

    apply();
    window.addEventListener('resize', apply, { passive: true });
}