// Minimal, safe service worker to satisfy PWA audits without console noise
// Strategy: network-first for HTML; cache-first for static assets

const SW_VERSION = 'v1';
const CORE_CACHE = `core-${SW_VERSION}`;
const ASSET_CACHE = `assets-${SW_VERSION}`;

// Only include URLs we are sure exist relative to the current scope
const CORE_FILES = [
  './policy.html',
  './styles.css',
  './script.js',
  './logo.png'
];

self.addEventListener('install', (event) => {
  event.waitUntil(
    caches.open(CORE_CACHE).then((cache) => cache.addAll(CORE_FILES)).catch(() => {})
  );
});

self.addEventListener('activate', (event) => {
  event.waitUntil(
    caches.keys().then((keys) =>
      Promise.all(
        keys
          .filter((k) => ![CORE_CACHE, ASSET_CACHE].includes(k))
          .map((k) => caches.delete(k))
      )
    ).catch(() => {})
  );
});

self.addEventListener('fetch', (event) => {
  const req = event.request;
  const url = new URL(req.url);

  // Only handle same-origin requests
  if (url.origin !== self.location.origin) return;

  // Network-first for navigation (HTML)
  if (req.mode === 'navigate') {
    event.respondWith(
      fetch(req).then((res) => {
        const copy = res.clone();
        caches.open(CORE_CACHE).then((c) => c.put(req, copy)).catch(() => {});
        return res;
      }).catch(() => caches.match(req))
    );
    return;
  }

  // Cache-first for static assets (CSS/JS/Images)
  if (/(\.css|\.js|\.png|\.jpg|\.jpeg|\.svg|\.ico)$/i.test(url.pathname)) {
    event.respondWith(
      caches.match(req).then((hit) => {
        if (hit) return hit;
        return fetch(req).then((res) => {
          const copy = res.clone();
          caches.open(ASSET_CACHE).then((c) => c.put(req, copy)).catch(() => {});
          return res;
        });
      }).catch(() => fetch(req))
    );
  }
});


